<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

require 'config.php';

$conn = new mysqli($servername, $username, $password, $dbname);

if ($conn->connect_error) {
    die(json_encode(["error" => "Database connection failed: " . $conn->connect_error]));
}

try {
    // Get query parameters
    $customer_id = $_GET['customer_id'] ?? '';
    $count_only = $_GET['count_only'] ?? '';
    $search = $_GET['search'] ?? '';
    $verification_type = $_GET['verification_type'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = max(1, min(100, intval($_GET['limit'] ?? 10)));
    $offset = ($page - 1) * $limit;

    // Build the base query with JOIN to get customer name
    $whereConditions = [];
    $params = [];
    $types = '';

    $baseQuery = "
        FROM login_attempts la 
        LEFT JOIN records r ON la.customer_id = r.id 
        WHERE 1=1
    ";

    // Add customer filter if specified
    if (!empty($customer_id)) {
        $whereConditions[] = "la.customer_id = ?";
        $params[] = intval($customer_id);
        $types .= 'i';
    }

    // Add search condition
    if (!empty($search)) {
        $whereConditions[] = "(la.mobile LIKE ? OR r.name LIKE ? OR la.user_id LIKE ?)";
        $searchTerm = "%$search%";
        $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm]);
        $types .= 'sss';
    }

    // Add verification type filter
    if (!empty($verification_type)) {
        $whereConditions[] = "la.verification_type = ?";
        $params[] = $verification_type;
        $types .= 's';
    }

    // Combine conditions
    if (!empty($whereConditions)) {
        $baseQuery .= " AND " . implode(" AND ", $whereConditions);
    }

    // If only count is requested
    if (!empty($count_only)) {
        $countQuery = "SELECT COUNT(*) as total " . $baseQuery;
        if (!empty($params)) {
            $countStmt = $conn->prepare($countQuery);
            if (!empty($types)) {
                $countStmt->bind_param($types, ...$params);
            }
            $countStmt->execute();
            $countResult = $countStmt->get_result();
            $total = $countResult->fetch_assoc()['total'];
            $countStmt->close();
        } else {
            $countResult = $conn->query($countQuery);
            $total = $countResult->fetch_assoc()['total'];
        }

        echo json_encode([
            "success" => true,
            "count" => intval($total)
        ]);
        $conn->close();
        return;
    }

    // Get total count
    $countQuery = "SELECT COUNT(*) as total " . $baseQuery;
    if (!empty($params)) {
        $countStmt = $conn->prepare($countQuery);
        if (!empty($types)) {
            $countStmt->bind_param($types, ...$params);
        }
        $countStmt->execute();
        $countResult = $countStmt->get_result();
        $total = $countResult->fetch_assoc()['total'];
        $countStmt->close();
    } else {
        $countResult = $conn->query($countQuery);
        $total = $countResult->fetch_assoc()['total'];
    }

    // Get the actual data
    $dataQuery = "
        SELECT 
            la.id,
            la.customer_id,
            la.mobile,
            la.verification_type,
            la.user_id,
            la.card_number,
            la.registered_mobile,
            la.ip_address,
            la.user_agent,
            la.attempt_time,
            la.status,
            r.name as customer_name
        " . $baseQuery . "
        ORDER BY la.attempt_time DESC 
        LIMIT ? OFFSET ?
    ";

    $params[] = $limit;
    $params[] = $offset;
    $types .= 'ii';

    if (!empty($params)) {
        $dataStmt = $conn->prepare($dataQuery);
        if (!empty($types)) {
            $dataStmt->bind_param($types, ...$params);
        }
        $dataStmt->execute();
        $result = $dataStmt->get_result();
    } else {
        $result = $conn->query($dataQuery);
    }

    $attempts = [];
    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $attempts[] = $row;
        }
    }

    // Close statement if it was prepared
    if (isset($dataStmt)) {
        $dataStmt->close();
    }

    // Return the response
    echo json_encode([
        "success" => true,
        "data" => $attempts,
        "pagination" => [
            "total" => intval($total),
            "page" => $page,
            "limit" => $limit,
            "pages" => ceil($total / $limit)
        ]
    ]);

} catch (Exception $e) {
    error_log("Error in fetchLoginAttempts.php: " . $e->getMessage());
    echo json_encode(["error" => "Database error: " . $e->getMessage()]);
}

$conn->close();
?>
